<?php

/**
 * Handles the installation process.
 *
 * @author     Rose
 * @since      2021
 */

use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\File;
use Illuminate\Support\Facades\DB;
use Illuminate\Http\Request;

require __DIR__ . '/../vendor/autoload.php';
$app = require_once __DIR__ . '/../bootstrap/app.php';
$kernel = $app->make(Illuminate\Contracts\Http\Kernel::class);
$kernel->handle(Illuminate\Http\Request::capture());

/**
 * Handle the installation request.
 */
function handleInstallation(Request $request)
{
    switch ($request->input('case')) {
        case 'environment':
            return handleEnvironmentSetup($request);

        case 'complete':
            return completeInstallation($request);

        default:
            return renderInstallView('environment');
    }
}

/**
 * Handle the environment setup.
 *
 * @param Request $request
 * @return string
 */
function handleEnvironmentSetup(Request $request)
{
    try {
        $domain = $request->getSchemeAndHttpHost();
        $envExamplePath = base_path('/.env.example');
        $envContent = file_get_contents($envExamplePath);

        if ($envContent === false) {
            throw new Exception("Unable to read .env.example file.");
        }

        $replacements = [
            'your_domain' => $domain,
            'database_hostname' => $request->input('database_hostname'),
            'database_port' => $request->input('database_port'),
            'database_name' => $request->input('database_name'),
            'database_username' => '"' . $request->input('database_username') . '"',
            'database_password' => '"' . $request->input('database_password') . '"',
            'bypass_storage_link' => $request->boolean('bypass_storage_link') ? 'true' : 'false',
        ];

        $envContent = str_replace(array_keys($replacements), array_values($replacements), $envContent);

        $envPath = base_path('/.env');
        if (file_put_contents($envPath, $envContent) === false) {
            throw new Exception("Failed to write to .env file.");
        }

        return renderInstallView('complete', ['message' => 'Database credentials saved successfully.']);
    } catch (Exception $e) {
        return renderInstallView('environment', ['error' => $e->getMessage()]);
    }
}

/**
 * Complete the installation process.
 *
 * @param Request $request
 * @return string
 */
function completeInstallation(Request $request)
{
    try {
        DB::connection()->getPdo();
        Artisan::call('install:initialize');

        return renderInstallView('environment', [
            'message' => 'Installation completed successfully.',
            'path' => $request->getSchemeAndHttpHost(),
            'output' => Artisan::output(),
        ]);
    } catch (Exception $e) {
        File::delete(base_path('/.env'));
        return renderInstallView('environment', ['error' => $e->getMessage()]);
    }
}

/**
 * Render the installation view.
 *
 * @param string $case
 * @param array $data
 * @return string
 */
function renderInstallView(string $case, array $data = [])
{
    return view('install', array_merge(['case' => $case], $data));
}

/**
 * Perform cleanup and redirect if already installed.
 */
function checkIfInstalled()
{
    if (File::exists(storage_path('installed')) && File::exists(base_path('/.env'))) {
        Artisan::call('install:update');
        // Clean up installation files
        File::delete(base_path('resources/views/install.blade.php'));
        File::delete(base_path('LocalValetDriver.php'));
        File::delete(public_path('install.php'));

        header("Location: /admin/dashboard");
        exit;
    }
}

// Main Execution
checkIfInstalled();
if (request()->method() === 'POST') {
    echo handleInstallation(request());
} else {
    echo renderInstallView('environment');
    exit;
}
