<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\ApiController;
use App\Http\Requests\GuestRoomReservationStoreRequest;
use App\Http\Resources\BookingSourceResource;
use App\Http\Resources\CustomerResource;
use App\Http\Resources\GuestRoomReservationDetailsResource;
use App\Http\Resources\GuestRoomReservationResource;
use App\Http\Resources\GuestRoomResource;
use App\Http\Resources\PosTaxResource;
use App\Http\Resources\ReservationStatusResource;
use App\Models\BookingSource;
use App\Models\Customer;
use App\Models\GuestRoom;
use App\Models\GuestRoomReservation;
use App\Models\ReservationStatus;
use App\Models\Tax;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Str;

class GuestRoomReservationController extends ApiController
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request): JsonResponse
    {
        $sort = $this->sort($request);
        $reservations = GuestRoomReservation::filter($request->all())
            ->advanceDuration($request->duration, $request->starting_date, $request->starting_time, $request->ending_date, $request->ending_time)
            ->orderBy($sort['column'], $sort['order'])
            ->paginate((int) $request->get('perPage', 10));
        return response()->json(
            [
                'items' => GuestRoomReservationResource::collection($reservations->items()),
                'pagination' => $this->pagination($reservations),
            ]
        );
    }


    /**
     * Store a newly created resource in storage.
     */
    public function store(GuestRoomReservationStoreRequest $request): JsonResponse
    {
        $validated = $request->validated();
        $validated['tracking'] = time();
        $validated['uuid'] = Str::orderedUuid();


        if ($request->checkin_process) {
            $validated['checkin_at'] = $this->getCurrentTimpstamp();
        }
        $guestRoomReservation  = GuestRoomReservation::create($validated);
        return response()->json([
            'message' => __('Data saved successfully'),
            'reservation' =>  $guestRoomReservation->uuid
        ]);
    }

    /**
     * Display the specified resource.
     */
    public function show(GuestRoomReservation $guestRoomReservation): JsonResponse
    {
        return response()->json(new GuestRoomReservationDetailsResource($guestRoomReservation));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(GuestRoomReservationStoreRequest $request, GuestRoomReservation $guestRoomReservation)
    {
        $validated = $request->validated();
        if ($request->checkin_process) {
            $validated['checkin_at'] = $this->getCurrentTimpstamp();
        }
        $guestRoomReservation->update($validated);
        return response()->json([
            'message' => __('Data updated successfully'),
        ]);
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(GuestRoomReservation $guestRoomReservation)
    {
        if (!Auth::user()->userRole->checkPermission('remove_sales')) {
            return response()->json([
                'message' => __('You have not permit to perform this request'),
            ], 403);
        }
        foreach ($guestRoomReservation->sales as $key => $sale) {
            \App\Helpers\ActivityLogger::log(__('Sale order removed successfully'), $sale->tracking);
            $sale->delete();
        }
        $guestRoomReservation->delete();
        return response()->json(['message' => __('Data removed successfully')]);
    }


    public function filters(): JsonResponse
    {
        return response()->json([
            'customers' => CustomerResource::collection(Customer::get()),
            'statuses' => ReservationStatusResource::collection(ReservationStatus::get()),
            'sources' => BookingSourceResource::collection(BookingSource::get()),
            'rooms' => GuestRoomResource::collection(GuestRoom::get()),
            'taxes' => PosTaxResource::collection(Tax::get()),
        ]);
    }
}
