<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\ApiController;
use App\Http\Requests\CashbookRequest;
use App\Http\Resources\CashBookResource;
use App\Models\CashBook;
use Auth;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Str;

class CashBookController extends ApiController
{

    /**
     * Display a listing of the resource.
     *
     * @param      \Illuminate\Http\Request  $request  The request
     *
     * @return     JsonResponse              The json response.
     */
    public function index(Request $request): JsonResponse
    {
        $sort = $this->sort($request);
        $list = CashBook::orderBy($sort['column'], $sort['order'])
            ->paginate((int) $request->get('perPage', 10));
        return response()->json(
            [
                'items' => CashBookResource::collection($list->items()),
                'pagination' => $this->pagination($list),
            ]
        );
    }

    /**
     * Display a listing of the resource.
     */
    public function current(): JsonResponse
    {
        return response()->json(
            $this->activeRegister()
                ? new CashBookResource($this->activeRegister())
                : []
        );
    }

    public function show(CashBook $cashBook): JsonResponse
    {
        return response()->json(new CashBookResource($cashBook));
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param      \App\Http\Requests\CashbookRequest  $request  The request
     *
     * @return     JsonResponse                        The json response.
     */
    public function store(CashbookRequest $request): JsonResponse
    {
        if ($this->getCurrentUserCashbook()) {
            return response()->json([
                'message' => __('You have already cash register opened'),
            ], 422);
        }
        $validated = $request->validated();
        $validated['uuid'] = Str::orderedUuid();
        $validated['user_id'] = Auth::user()->id;
        $item = CashBook::create($validated);
        \App\Helpers\ActivityLogger::log(__('Cashbook register opened successfully'));
        return response()->json([
            'message' => __('Cash register opened successfully'),
            'cashbook' => new CashBookResource($item),
        ]);
    }

    /**
     * Updates the given cash book.
     *
     * @param      \App\Models\CashBook  $cashBook  The cash book
     *
     * @return     JsonResponse          The json response.
     */
    public function update(CashBook $cashBook): JsonResponse
    {
        if ($cashBook->canBeClosed()) {
            $cashBook->update(['closed' => true]);
            \App\Helpers\ActivityLogger::log(__('Cashbook register closed successfully'));
            return response()->json([
                'message' => __('Register has been closed'),
            ]);
        }
        return response()->json([
            'message' => __('Please clear all orders in progress before closing the register'),
        ], 422);
    }
}
